<?php

class XenGallery_DataWriter_Album extends XenForo_DataWriter
{
	/**
	* Holds the reason for soft deletion.
	*
	* @var string
	*/
	const DATA_DELETE_REASON = 'deleteReason';

	/**
	* Title of the phrase that will be created when a call to set the
	* existing data fails (when the data doesn't exist).
	*
	* @var string
	*/
	protected $_existingDataErrorPhrase = 'xengallery_requested_album_not_found';

	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xengallery_album' => array(
				'album_id' => array('type' => self::TYPE_UINT, 'autoIncrement' => true),
				'album_title' => array('type' => self::TYPE_STRING, 'maxLength' => 50, 'required' => true,
					'requiredError' => 'please_enter_valid_title'
				),
				'album_description' => array('type' => self::TYPE_STRING, 'default' => '', 'maxLength' => 200),
				'album_create_date' => array('type' => self::TYPE_UINT, 'default' => XenForo_Application::$time),
				'last_update_date' => array('type' => self::TYPE_UINT, 'default' => '0'),
				'random_media_cache' => array('type' => self::TYPE_SERIALIZED),
				'album_share_users' => array('type' => self::TYPE_SERIALIZED),
				'album_state' => array('type' => self::TYPE_STRING, 'default' => 'visible',
					'allowedValues' => array('visible', 'moderated', 'deleted')
				),
				'album_privacy' => array('type' => self::TYPE_STRING, 'default' => 'private',
					'allowedValues' => array('private', 'public', 'shared')
				),
				'user_id' => array('type' => self::TYPE_UINT, 'default' => 0),
				'username' => array('type' => self::TYPE_STRING, 'maxLength' => 50),
				'ip_id' => array('type' => self::TYPE_UINT, 'default' => 0),
				'album_likes' => array('type' => self::TYPE_UINT),
				'album_like_users' => array('type' => self::TYPE_SERIALIZED),
				'album_media_count' => array ('type' => self::TYPE_UINT, 'default' => 0),
				'album_view_count' => array('type' => self::TYPE_UINT, 'default' => 0),				
				'album_rating_count' => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),
				'album_rating_sum' => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),
				'album_rating_avg' => array('type' => self::TYPE_FLOAT, 'default' => 0)
			)
		);
	}
	
	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$id = $this->_getExistingPrimaryKey($data))
		{
			return false;
		}

		return array('xengallery_album' => $this->_getAlbumModel()->getAlbumById($id));
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'album_id = ' . $this->_db->quote($this->getExisting('album_id'));
	}
	
	protected function _preSave()
	{
		if ($this->isUpdate())
		{
			$this->set('last_update_date', XenForo_Application::$time);
		}
		
		$db = $this->_db;
		
		if ($this->isChanged('album_privacy'))
		{
			if ($this->get('album_privacy') == 'private')
			{
				$db->insert('xengallery_private_map', array('album_id' => $this->get('album_id'), 'private_user_id' => $this->get('user_id')));
				$db->delete('xengallery_shared_map', 'album_id = ' . $this->get('album_id'));
			}
			
			if ($this->getExisting('album_privacy') == 'private')
			{
				$db->delete('xengallery_private_map', 'album_id = ' . $this->get('album_id') . ' AND private_user_id = ' . $this->get('user_id'));
			}			
		}
	}
	
	protected function _postSave()
	{
		if ($this->isInsert())
		{
			$ipId = XenForo_Model_Ip::log(
				$this->get('user_id'), 'xengallery_album', $this->get('album_id'), 'insert'
			);
			
			$this->_db->update('xengallery_album', array(
				'ip_id' => $ipId
			), 'album_id = ' . $this->get('album_id'));
			
			$this->updateAlbumMediaCount();
		}
		
		if ($this->isChanged('album_privacy'))
		{
			$this->_db->update('xengallery_media', array(
				'media_privacy' => $this->get('album_privacy')
			), 'album_id = ' . $this->get('album_id'));
		}
	}
	
	/**
	 * Post-delete handling.
	 */
	protected function _postDelete()
	{
		if ($this->getExisting('album_state') == 'visible')
		{
			$this->updateAlbumMediaCount(false);
		}
	}
	
	public function updateRating($adjustSum = null, $adjustCount = null)
	{
		if ($adjustSum === null && $adjustCount === null)
		{
			$rating = $this->_db->fetchRow("
				SELECT COUNT(*) AS total, SUM(rating) AS sum
				FROM xengallery_rating
				WHERE content_id = ?
				AND content_type = 'album'
				", $this->get('album_id'));
	
			$this->set('album_rating_sum', $rating['sum']);
			$this->set('album_rating_count', $rating['total']);
		}
		else
		{
			if ($adjustSum !== null)
			{
				$this->set('album_rating_sum', $this->get('album_rating_sum') + $adjustSum);
			}
			if ($adjustCount !== null)
			{
				$this->set('album_rating_count', $this->get('album_rating_count') + $adjustCount);
			}
		}
	
		if ($this->get('album_rating_count'))
		{
			$this->set('album_rating_avg', $this->get('album_rating_sum') / $this->get('album_rating_count'));
		}
		else
		{
			$this->set('album_rating_avg', 0);
		}
	}
	
	public function updateAlbumMediaCount($increase = true)
	{
		$value = '+ 1';
	
		if (!$increase)
		{
			$value = '- 1';
		}
	
		$this->_db->query("
			UPDATE xf_user
			SET album_count = album_count $value
			WHERE user_id = ?
		", $this->get('user_id'));
	}

	protected function _getMediaModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Media');
	}
	
	protected function _getAlbumModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Album');
	}
}
