<?php

class XenResource_Model_Category extends XenForo_Model
{
	public function getCategoryById($categoryId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xf_resource_category
			WHERE resource_category_id = ?
		', $categoryId);
	}

	public function getCategoriesByIds(array $categoryIds)
	{
		if (!$categoryIds)
		{
			return array();
		}

		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			WHERE resource_category_id IN (' . $this->_getDb()->quote($categoryIds) . ')
		', 'resource_category_id');
	}

	public function getAllCategories()
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			ORDER BY lft
		', 'resource_category_id');
	}

	public function getViewableCategories()
	{
		$categories = $this->getAllCategories();
		if (!$categories)
		{
			return array();
		}

		foreach ($categories AS $key => $category)
		{
			if (!$this->canViewCategory($category))
			{
				unset($categories[$key]);
			}
		}

		return $categories;
	}

	public function getChildrenOfCategoryId($categoryId)
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			WHERE parent_category_id = ?
			ORDER BY lft
		', 'resource_category_id', $categoryId);
	}

	public function getDescendantsOfCategory($category, $rgt = null)
	{
		if (is_array($category))
		{
			$lft = $category['lft'];
			$rgt = $category['rgt'];
		}
		else if (!is_null($rgt))
		{
			$lft = intval($category);
		}
		else if (!$category)
		{
			return $this->getAllCategories();
		}
		else
		{
			$category = $this->getCategoryById($category);
			if (!$category)
			{
				return array();
			}

			$lft = $category['lft'];
			$rgt = $category['rgt'];
		}

		if ($rgt == $lft + 1)
		{
			return array();
		}

		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			WHERE lft > ? AND rgt < ?
			ORDER BY lft
		', 'resource_category_id', array($lft, $rgt));
	}

	public function getDescendantsOfCategoryIds(array $categoryIds)
	{
		$categories = $this->getAllCategories();

		$ranges = array();
		foreach ($categoryIds AS $categoryId)
		{
			if (isset($categories[$categoryId]))
			{
				$category = $categories[$categoryId];
				$ranges[] = array($category['lft'], $category['rgt']);
			}
		}

		$descendants = array();
		foreach ($categories AS $category)
		{
			foreach ($ranges AS $range)
			{
				if ($category['lft'] > $range[0] && $category['lft'] < $range[1])
				{
					$descendants[$category['resource_category_id']] = $category;
					break;
				}
			}
		}

		return $descendants;
	}

	public function getPossibleParentCategories(array $category)
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			WHERE lft < ? OR rgt > ?
			ORDER BY lft
		', 'resource_category_id', array($category['lft'], $category['rgt']));
	}

	public function groupCategoriesByParent(array $categories)
	{
		$grouped = array();
		foreach ($categories AS $category)
		{
			$grouped[$category['parent_category_id']][$category['resource_category_id']] = $category;
		}

		return $grouped;
	}

	public function getDescendantCategoryIdsFromGrouped(array $grouped, $parentCategoryId = 0)
	{
		$parentIds = array($parentCategoryId);
		$output = array();
		do
		{
			$parentId = array_shift($parentIds);
			if (isset($grouped[$parentId]))
			{
				$keys = array_keys($grouped[$parentId]);
				$output = array_merge($output, $keys);
				$parentIds = array_merge($parentIds, $keys);
			}
		}
		while ($parentIds);

		return $output;
	}

	public function getCategoryBreadcrumb(array $category, $includeSelf = true)
	{
		$breadcrumbs = array();

		if (!isset($category['categoryBreadcrumb']))
		{
			$category['categoryBreadcrumb'] = unserialize($category['category_breadcrumb']);
		}

		foreach ($category['categoryBreadcrumb'] AS $catId => $breadcrumb)
		{
			$breadcrumbs[$catId] = array(
				'href' => XenForo_Link::buildPublicLink('full:resources/categories', $breadcrumb),
				'value' => $breadcrumb['category_title']
			);
		}

		if ($includeSelf)
		{
			$breadcrumbs[$category['resource_category_id']] = array(
				'href' => XenForo_Link::buildPublicLink('full:resources/categories', $category),
				'value' => $category['category_title']
			);
		}

		return $breadcrumbs;
	}

	/**
	 * Gets permission-based conditions that apply to resource fetching functions.
	 *
	 * @param array|null $category Category the resources will belong to
	 * @param array|null $viewingUser
	 *
	 * @return array Keys: deleted (boolean), moderated (boolean or integer, if can only view single user's)
	 */
	public function getPermissionBasedFetchConditions(array $category = null, array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'resource', 'viewModerated'))
		{
			$viewModerated = true;
		}
		else if ($viewingUser['user_id'])
		{
			$viewModerated = $viewingUser['user_id'];
		}
		else
		{
			$viewModerated = false;
		}

		$conditions = array(
			'deleted' => XenForo_Permission::hasPermission($viewingUser['permissions'], 'resource', 'viewDeleted'),
			'moderated' => $viewModerated
		);

		return $conditions;
	}

	public function prepareCategory(array $category)
	{
		$category['categoryBreadcrumb'] = unserialize($category['category_breadcrumb']);
		$category['last_resource_title'] = XenForo_Helper_String::censorString($category['last_resource_title']);
		$category['allowResource'] = (
			$category['allow_local']
			|| $category['allow_external']
			|| $category['allow_commercial_external']
			|| $category['allow_fileless']
		);
		$category['canAdd'] = $this->canAddResource($category);

		return $category;
	}

	public function prepareCategories(array $categories)
	{
		foreach ($categories AS &$category)
		{
			$category = $this->prepareCategory($category);
		}

		return $categories;
	}

	public function rebuildCategoryStructure()
	{
		$grouped = $this->groupCategoriesByParent($this->fetchAllKeyed('
			SELECT *
			FROM xf_resource_category
			ORDER BY display_order
		', 'resource_category_id'));

		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);

		$changes = $this->_getStructureChanges($grouped);
		foreach ($changes AS $categoryId => $change)
		{
			$db->update('xf_resource_category', $change, 'resource_category_id = ' . $db->quote($categoryId));
		}

		XenForo_Db::commit($db);

		return $changes;
	}

	protected function _getStructureChanges(array $grouped, $parentId = 0, $depth = 0,
		$startPosition = 1, &$nextPosition = 0, array $breadcrumb = array()
	)
	{
		$nextPosition = $startPosition;

		if (!isset($grouped[$parentId]))
		{
			return array();
		}

		$changes = array();
		$serializedBreadcrumb = serialize($breadcrumb);

		foreach ($grouped[$parentId] AS $categoryId => $category)
		{
			$left = $nextPosition;
			$nextPosition++;

			$thisBreadcrumb = $breadcrumb + array(
				$categoryId => array(
					'resource_category_id' => $categoryId,
					'category_title' => $category['category_title'],
					'parent_category_id' => $category['parent_category_id'],
					'depth' => $category['depth'],
					'lft' => $category['lft'],
					'rgt' => $category['rgt'],
				)
			);

			$changes += $this->_getStructureChanges(
				$grouped, $categoryId, $depth + 1, $nextPosition, $nextPosition, $thisBreadcrumb
			);

			$catChanges = array();
			if ($category['depth'] != $depth)
			{
				$catChanges['depth'] = $depth;
			}
			if ($category['lft'] != $left)
			{
				$catChanges['lft'] = $left;
			}
			if ($category['rgt'] != $nextPosition)
			{
				$catChanges['rgt'] = $nextPosition;
			}
			if ($category['category_breadcrumb'] != $serializedBreadcrumb)
			{
				$catChanges['category_breadcrumb'] = $serializedBreadcrumb;
			}

			if ($catChanges)
			{
				$changes[$categoryId] = $catChanges;
			}

			$nextPosition++;
		}

		return $changes;
	}

	public function applyRecursiveCountsToGrouped(array $grouped, $parentCategoryId = 0)
	{
		if (!isset($grouped[$parentCategoryId]))
		{
			return array();
		}

		$this->_applyRecursiveCountsToGrouped($grouped, $parentCategoryId);
		return $grouped;
	}

	protected function _applyRecursiveCountsToGrouped(array &$grouped, $parentCategoryId)
	{
		$output = array(
			'resource_count' => 0,
			'last_update' => 0
		);

		foreach ($grouped[$parentCategoryId] AS $categoryId => &$category)
		{
			if (isset($grouped[$categoryId]))
			{
				$childCounts = $this->_applyRecursiveCountsToGrouped($grouped, $categoryId);

				$category['resource_count'] += $childCounts['resource_count'];
				if ($childCounts['last_update'] > $category['last_update'])
				{
					$category['last_update'] = $childCounts['last_update'];
					$category['last_resource_title'] = $childCounts['last_resource_title'];
					$category['last_resource_id'] = $childCounts['last_resource_id'];
				}
			}

			$output['resource_count'] += $category['resource_count'];
			if ($category['last_update'] > $output['last_update'])
			{
				$output['last_update'] = $category['last_update'];
				$output['last_resource_title'] = $category['last_resource_title'];
				$output['last_resource_id'] = $category['last_resource_id'];
			}
		}

		return $output;
	}

	/**
	 * Determines if a user can view a given resource category.
	 *
	 * @param array $category
	 * @param string $errorPhraseKey
	 * @param array $viewingUser
	 *
	 * @return boolean
	 */
	public function canViewCategory(array $category, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'resource', 'view');
	}

	/**
	 * Determines if a user can add a resource. Does not check category viewing perms.
	 *
	 * @param array|null $category May be null if no category is specified
	 * @param string $errorPhraseKey
	 * @param array $viewingUser
	 *
	 * @return boolean
	 */
	public function canAddResource(array $category = null, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($category)
		{
			if (!isset($category['allowResource']))
			{
				$category['allowResource'] = (
					$category['allow_local']
					|| $category['allow_external']
					|| $category['allow_commercial_external']
					|| $category['allow_fileless']
				);
			}

			if (!$category['allowResource'])
			{
				$errorPhraseKey = 'category_not_allow_new_resources';
				return false;
			}

			if (!XenForo_Permission::hasPermission($viewingUser['permissions'], 'resource', 'add'))
			{
				return false;
			}

			$userGroups = explode(',', $category['allow_submit_user_group_ids']);
			if (in_array(-1, $userGroups) || in_array($viewingUser['user_group_id'], $userGroups))
			{
				return true;
			}

			if ($viewingUser['secondary_group_ids'])
			{
				foreach (explode(',', $viewingUser['secondary_group_ids']) AS $userGroupId)
				{
					if (in_array($userGroupId, $userGroups))
					{
						return true;
					}
				}
			}

			return false;
		}
		else
		{
			return XenForo_Permission::hasPermission($viewingUser['permissions'], 'resource', 'add');
		}
	}
}