<?php

class XenGallery_Model_Album extends XenForo_Model
{
	const FETCH_USER = 0x01;
	const FETCH_USER_OPTION = 0x02;	
	
	public static $voteThreshold = 10;
	public static $averageVote = 3;
		
	/**
	 * Gets a single album record specified by its ID
	 *
	 * @param integer $albumId
	 *
	 * @return array
	 */
	public function getAlbumById($albumId, array $fetchOptions = array())
	{
		$joinOptions = $this->prepareAlbumFetchOptions($fetchOptions);
		
		return $this->_getDb()->fetchRow('
			SELECT album.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_album AS album
				' . $joinOptions['joinTables'] . '
			WHERE album.album_id = ?
		', $albumId);
	}
	
	/**
	 * Gets album records specified by their IDs
	 *
	 * @param array $albumIds
	 *
	 * @return array
	 */
	public function getAlbumsByIds($albumIds, array $fetchOptions = array())
	{
		$db = $this->_getDb();
		$joinOptions = $this->prepareAlbumFetchOptions($fetchOptions);

		return $this->fetchAllKeyed('
			SELECT album.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_album AS album
				' . $joinOptions['joinTables'] . '
			WHERE album.album_id IN (' . $db->quote($albumIds) . ')
		', 'album_id');
	}	
	
	/**
	 * Gets albums based on various criteria
	 *
	 * @param array $condtions
	 * @param array $fetchOptions 
	 *
	 * @return array
	 */
	public function getAlbums(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareAlbumConditions($conditions, $fetchOptions);
		
		$joinOptions = $this->prepareAlbumFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		$sqlClauses = $this->prepareAlbumFetchOptions($fetchOptions);

		$albums = $this->fetchAllKeyed($this->limitQueryResults('
			SELECT album.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_album AS album
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				' . $sqlClauses['orderClause'] . '			
			', $limitOptions['limit'], $limitOptions['offset']
		), 'album_id');
		
		return $albums;
	}
	
	public function getSharedAlbums(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareAlbumConditions($conditions, $fetchOptions);
		
		$joinOptions = $this->prepareAlbumFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		$sqlClauses = $this->prepareAlbumFetchOptions($fetchOptions);
		
		$albums = $this->fetchAllKeyed($this->limitQueryResults('
			SELECT shared.*, album.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_shared_map AS shared
			INNER JOIN xengallery_album AS album ON
				(album.album_id = shared.album_id)
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				' . $sqlClauses['orderClause'] . '			
			', $limitOptions['limit'], $limitOptions['offset']
		), 'album_id');
		
		return $albums;
	}
	
	/**
	 * Gets a list of media Ids based on the album Id
	 * 
	 * @param int $albumId
	 * 
	 * @return array
	 */
	public function getMediaIdsFromAlbum($albumId)
	{
		return $this->_getDb()->fetchCol('
			SELECT media_id
			FROM xengallery_media
			WHERE album_id = ?
		', $albumId);
	}
	
	public function countMedia(array $conditions = array())
	{	
		$fetchOptions = array();
		$whereClause = $this->prepareMediaConditions($conditions, $fetchOptions);

		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);
				
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xengallery_media AS media
			' . $joinOptions['joinTables'] . '
			WHERE ' . $whereClause
		);
	}	
	
	public function getTopContributors($limit)
	{
		return $this->_getDb()->fetchAll('
			SELECT user.user_id, user.username,
				user.media_count, user.avatar_date,
				user.gravatar, user.avatar_width,
				user.avatar_height, user.display_style_group_id
			FROM xf_user AS user
			WHERE media_count > 0
			ORDER BY media_count DESC
			LIMIT ?
		', $limit);
	}
	
	/**
	 * Gets the average rating based on the sum and count stored.
	 *
	 * @param integer $sum
	 * @param integer $count
	 * @param boolean $round If true, return rating to the nearest 0.5, otherwise full float.
	 *
	 * @return float
	 */
	public function getRatingAverage($sum, $count, $round = false)
	{
		if ($count == 0)
		{
			return 0;
		}

		$average = $sum / $count;

		if ($round)
		{
			$average = round($average / 0.5, 0) * 0.5;
		}

		return $average;
	}
	
	/**
	 * Shares an album and rebuilds the permissions cache
	 * 
	 * @param array $users
	 * @param array $album
	 * @param string $type
	 */
	public function share(array $users, array $album, $type = 'read')
	{
		$db = $this->_getDb();
		
		XenForo_Db::beginTransaction($db);
		
		$shareUsers = unserialize($album['album_share_users']);
		foreach ($users AS $userId => $user)
		{
			$data = array(
				'shared_user_id' => $userId,
				'album_id' => $album['album_id']
			);
			
			$db->insert('xengallery_shared_map', $data);
			
			$shareUsers[$userId] = $data;
		}
		
		$db->update('xengallery_album', array('album_share_users' => serialize($shareUsers)), 'album_id = ' . $album['album_id']);
		
		XenForo_Db::commit($db);
	}
	
	/**
	 * Unshares an album and rebuilds the permissions cache
	 * 
	 * @param int $userId
	 * @param array $albumId
	 */
	public function unshare($userId, array $album)
	{
		$db = $this->_getDb();
		
		XenForo_Db::beginTransaction($db);
		$shareUsers = unserialize($album['album_share_users']);
		
		if ($shareUsers !== false)
		{
			$db->delete('xengallery_shared_map', "album_id = $album[album_id] AND shared_user_id = $userId");
			
			unset($shareUsers[$userId]);
			
			$db->update('xengallery_album', array('album_share_users' => serialize($shareUsers)), 'album_id = ' . $album['album_id']);
		}
		
		XenForo_Db::commit($db);
	}
	
	/**
	 * Logs the viewing of an album.
	 *
	 * @param integer $albumId
	 */
	public function logAlbumView($albumId)
	{
		$this->_getDb()->query('
			INSERT ' . (XenForo_Application::get('options')->enableInsertDelayed ? 'DELAYED' : '') . ' INTO xengallery_album_view
				(album_id)
			VALUES
				(?)
		', $albumId);
	}

	/**
	 * Updates album views in bulk.
	 */
	public function updateAlbumViews()
	{
		$db = $this->_getDb();

		$updates = $db->fetchPairs('
			SELECT album_id, COUNT(*)
			FROM xengallery_album_view
			GROUP BY album_id
		');

		XenForo_Db::beginTransaction($db);

		$db->query('TRUNCATE TABLE xengallery_album_view');

		foreach ($updates AS $albumId => $views)
		{
			$db->query('
				UPDATE xengallery_album SET
					album_view_count = album_view_count + ?
				WHERE album_id = ?
			', array($views, $albumId));
		}

		XenForo_Db::commit($db);
	}
	
	
	public function prepareAlbum($album)
	{
		$visitor = XenForo_Visitor::getInstance();

		if (!empty($album['album_id']))
		{
			$album['album_title'] = XenForo_Helper_String::censorString($album['album_title']);
			$album['album_description'] = XenForo_Helper_String::censorString($album['album_description']);			
		}
		else
		{
			foreach ($album AS &$item)
			{	
				$item['isIgnored'] = array_key_exists($item['user_id'], $visitor->ignoredUsers);
				$item['album_title'] = XenForo_Helper_String::censorString($item['album_title']);
				$item['album_description'] = XenForo_Helper_String::censorString($item['album_description']);
			}			
		}

		return $album;
	}	

	/**
	 * Prepares join-related fetch options.
	 *
	 * @param array $fetchOptions
	 *
	 * @return array Containing 'selectFields' and 'joinTables' keys.
	 */
	public function prepareAlbumFetchOptions(array $fetchOptions)
	{
				
		$selectFields = '';
		$joinTables = '';		
		$orderBy = '';
		
		if (!empty($fetchOptions['order']))
		{
			$orderBySecondary = '';

			switch ($fetchOptions['order'])
			{
				case 'rand':
					$orderBy = 'RAND()';
					break;
					
				case 'album_create_date':
				case 'new':
				default:
					$orderBy = 'album.album_create_date';
					break;
					
				case 'album_id':
					$orderBy = 'album.album_create_date';
					$orderBySecondary = ', album.album_id DESC';
					break;
					
				case 'rating_avg':
					$orderBy = 'album.album_rating_avg';
					$orderBySecondary = ', album.album_create_date DESC';
					break;
					
				case 'view_count':
					$orderBy = 'album.album_view_count';
					$orderBySecondary = ', album.album_create_date DESC';
					break;
					
				case 'media_count':
					$orderBy = 'album.album_media_count';
					$orderBySecondary = ', album.album_media_count DESC';
					break;
					
				case 'rating_count':
					$orderBy = 'album.album_rating_count';
					$orderBySecondary = ', album.album_create_date DESC';
					break;											
					
				case 'likes':
					$orderBy = 'album.album_likes';
					$orderBySecondary = ', album.album_create_date DESC';
					break;										
			}
			if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'desc')
			{
				$orderBy .= ' DESC';
			}
			else
			{
				$orderBy .= ' ASC';
			}
			
			$orderBy .= $orderBySecondary;
		}

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_USER)
			{
				$selectFields .= ',
					user.*, user_profile.*, IF(user.username IS NULL, album.username, user.username) AS username';
				$joinTables .= '
					LEFT JOIN xf_user AS user ON
						(user.user_id = album.user_id)
					LEFT JOIN xf_user_profile AS user_profile ON
						(user_profile.user_id = album.user_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_USER_OPTION)
			{
				$selectFields .= ',
					user_option.*';
				$joinTables .= '
					LEFT JOIN xf_user_option AS user_option ON
						(user_option.user_id = album.user_id)';
			}
		}

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause' => ($orderBy ? "ORDER BY $orderBy" : '')
		);
	}
	
	/**
	 * Prepares a set of conditions against which to select albums.
	 *
	 * @param array $conditions List of conditions.
	 * @param array $fetchOptions The fetch options that have been provided. May be edited if criteria requires.
	 *
	 * @return string Criteria as SQL for where clause
	 */
	public function prepareAlbumConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'album.user_id IN (' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'album.user_id = ' . $db->quote($conditions['user_id']);
			}
		}
		
		if (!empty($conditions['shared_user_id']))
		{
			if (is_array($conditions['shared_user_id']))
			{
				$sqlConditions[] = 'shared.shared_user_id IN (' . $db->quote($conditions['shared_user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'shared.shared_user_id = ' . $db->quote($conditions['shared_user_id']);
			}
		}

		if (!empty($conditions['album_id']))
		{
			if (is_array($conditions['album_id']))
			{
				$sqlConditions[] = 'album.album_id IN (' . $db->quote($conditions['album_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'album.album_id = ' . $db->quote($conditions['album_id']);
			}
		}

		return $this->getConditionsForClause($sqlConditions);
	}
	
	public function canViewAlbums(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewAlbums'))
		{
			return true;
		}		
		
		$errorPhraseKey = 'xengallery_no_album_view_permission';
		return false;
	}
	
	public function canViewAlbum(array $album, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewOverride'))
		{
			return true;
		}
		
		if ($album['album_privacy'] == 'public')
		{
			return true;
		}
		
		if ($album['album_privacy'] == 'private' && $viewingUser['user_id'] == $album['user_id'])
		{
			return true;
		}
		
		if ($album['album_privacy'] == 'shared')
		{
			if ($viewingUser['user_id'] == $album['user_id'])
			{
				return true;
			}
			
			$allowedUsers = unserialize($album['album_share_users']);
			
			if (!is_array($allowedUsers))
			{
				$allowedUsers = array();
			}
			
			if (isset($allowedUsers[$viewingUser['user_id']]))
			{
				return true;
			}
		}
		
		$errorPhraseKey = 'xengallery_no_view_this_album_permission';
		return false;		
	}
	
	public function canCreateAlbum(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
	
		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'createAlbum'))
		{
			return true;
		}
	
		$errorPhraseKey = 'xengallery_no_album_create_permission';
		return false;
	}
	
	public function canLikeAlbum(array $album, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
	
		if ($album['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_you_cannot_like_your_own_album';
			return false;
		}
	
		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'likeAlbum'))
		{
			return true;
		}
	
		$errorPhraseKey = 'xengallery_no_like_permission';
		return false;
	}
	
	public function canViewDeletedMedia(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewDeleted'))
		{
			return true;
		}
	
		return false;		
	}	
	
	public function canLikeMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_you_cannot_like_your_own_media';
			return false;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'like'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_like_permission';
		return false;		
	}
	
	public function canDeleteMedia(array $media, $type = 'soft', &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($type == 'hard')
		{
			return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'hardDeleteAny');
		}

		if ($media['user_id'] == $viewingUser['user_id'])
		{
			return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'delete');
		}

		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny');
	}
	
	public function canEditMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'edit'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_edit_permission';
		return false;		
	}
	
	public function canEditCaption(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCaption'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCaptionAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_caption_edit_permission';
		return false;		
	}	
	
	public function canApproveMedia(array $media, &$errorPhraseKey ='', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $media['media_state'] != 'moderated')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}	
	
	public function canUnapproveMedia(array $media, &$errorPhraseKey ='', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $media['media_state'] != 'visible')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}
	
	public function canViewUnapprovedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}		
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}	
	
	public function canRateAlbum(array $album, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($album['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_no_rate_album_by_self';
			return false;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'rateAlbum'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_rate_album_permission';
		return false;
	}
	
	public function canShareAlbum(array $album, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($album['album_privacy'] != 'shared')
		{
			$errorPhraseKey = 'xengallery_cannot_share_an_album_that_not_shared';
			return false;
		}
	
		if ($album['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'shareAlbum'))
		{
			return true;
		}
	
		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'shareAlbumAny'))
		{
			return true;
		}
	
		$errorPhraseKey = 'xengallery_no_share_album_permission';
		return false;
	}
	
	public function canAddMediaToCategory(array $allowedUserGroups, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$this->canAddMedia($errorPhraseKey, $viewingUser))
		{
			$errorPhraseKey = 'xengallery_you_cannot_add_media_to_this_category';
			return false;
		}
		
		if (in_array(-1, $allowedUserGroups))
		{
			return true;
		}
		
		$userModel = $this->getModelFromCache('XenForo_Model_User');
		foreach($allowedUserGroups AS $userGroupId)
		{
			if ($userModel->isMemberOfUserGroup($viewingUser, $userGroupId))
			{
				return true;
			}
		}
		
		$errorPhraseKey = 'xengallery_you_cannot_add_media_to_this_category';
		return false;
	}
	
	/**
	* Checks that the viewing user may managed a reported media item
	*
	* @param array $media
	* @param string $errorPhraseKey
	* @param array $viewingUser
	*
	* @return boolean
	*/
	public function canManageReportedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		return (
			XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny')
		);
	}
	
	/**
	* Checks that the viewing user may manage a moderated media item
	*
	* @param array $media
	* @param string $errorPhraseKey
	* @param array $viewingUser
	*
	* @return boolean
	*/
	public function canManageModeratedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		return (
			XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove')			
		);
	}
}
