<?php
 
class XenGallery_DataWriter_Media extends XenForo_DataWriter
{
	/**
	* Holds the reason for soft deletion.
	*
	* @var string
	*/
	const DATA_DELETE_REASON = 'deleteReason';

	/**
	* Title of the phrase that will be created when a call to set the
	* existing data fails (when the data doesn't exist).
	*
	* @var string
	*/
	protected $_existingDataErrorPhrase = 'xengallery_requested_media_not_found';

	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xengallery_media' => array(
				'media_id' => array('type' => self::TYPE_UINT, 'autoIncrement' => true),
				'media_title' => array('type' => self::TYPE_STRING, 'maxLength' => 50, 'required' => true,
					'requiredError' => 'please_enter_valid_title'
				),
				'media_description' => array('type' => self::TYPE_STRING, 'default' => '', 'maxLength' => 200),
				'media_caption' => array('type' => self::TYPE_STRING, 'default' => ''),
				'media_date' => array('type' => self::TYPE_UINT, 'default' => 0),
				'last_comment_date' => array('type' => self::TYPE_UINT, 'default' => 0),				
				'media_type' => array('type' => self::TYPE_STRING, 'default' => 'image_upload',
					'allowedValues' => array('image_upload', 'video_embed')
				),
				'media_tag' => array('type' => self::TYPE_STRING),
				'media_embed_url' => array('type' => self::TYPE_STRING),                
				'media_state' => array('type' => self::TYPE_STRING, 'default' => 'visible',
					'allowedValues' => array('visible', 'moderated', 'deleted')
				),
				'xengallery_category_id' => array('type' => self::TYPE_UINT, 'default' => 0),
				'attachment_id' => array('type' => self::TYPE_UINT, 'default' => 0),
				'user_id' => array('type' => self::TYPE_UINT, 'default' => 0),
				'username' => array('type' => self::TYPE_STRING, 'maxLength' => 50),
				'likes' => array('type' => self::TYPE_UINT),
				'like_users' => array('type' => self::TYPE_SERIALIZED),
				'comment_count' => array('type' => self::TYPE_UINT),
				'rating_count' => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),
				'rating_sum' => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),
				'rating_avg' => array('type' => self::TYPE_FLOAT, 'default' => 0)
			)
		);
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$id = $this->_getExistingPrimaryKey($data))
		{
			return false;
		}

		return array('xengallery_media' => $this->_getMediaModel()->getMediaById($id));
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'media_id = ' . $this->_db->quote($this->getExisting('media_id'));
	}

	protected function _preSave()
	{
		if ($this->isInsert())
		{
			$this->updateRating(
				intval($this->get('rating_sum')), intval($this->get('rating_count'))
			);
		}

		if ($this->isUpdate() && $this->isChanged('xengallery_category_id'))
		{
			$categoryModel = $this->_getCategoryModel();
			$destinationCategory = $categoryModel->getCategoryById($this->get('xengallery_category_id'));

			$allowedTypes = unserialize($destinationCategory['allowed_types']);

			$mediaType = $this->get('media_type');
			
			if (in_array('all', $allowedTypes))
			{
				return true;
			}

			if (!in_array($mediaType, $allowedTypes))
			{
				if ($mediaType == 'image_upload')
				{
					$this->error(new XenForo_Phrase('xengallery_you_cannot_put_images_in_this_category'));
				}
				
				if ($mediaType == 'video_embed')
				{
					$this->error(new XenForo_Phrase('xengallery_you_cannot_put_videos_in_this_category'));
				}
			}
		}
	}

	protected function _postSave()
	{
		if ($this->isInsert())
		{
			$this->updateUserMediaCount();

			$this->_getNewsFeedModel()->publish(
				$this->get('user_id'),
				$this->get('username'),
				'xengallery_media',
				$this->get('media_id'),
				'insert'
			);

			$ipId = XenForo_Model_Ip::log(
				$this->get('user_id'), 'xengallery_media', $this->get('media_id'), 'insert'
			);

			$this->_db->update('xengallery_media', array(
				'ip_id' => $ipId
				), 'media_id = ' . $this->get('media_id'));
				
			$this->_getMediaModel()->markMediaViewed(array('media_id' => $this->get('media_id')));
		}

		$media = $this->_getMediaModel()->getMediaById($this->get('media_id'), array());		
		
		$indexer = new XenForo_Search_Indexer();
		$dataHandler = XenForo_Search_DataHandler_Abstract::create('XenGallery_Search_DataHandler_Media');

		$dataHandler->insertIntoIndex($indexer, $this->getMergedData(), $media);
			
		$this->_updateDeletionLog();
		
		if ($this->get('media_state') == 'deleted')
		{
			$indexer = new XenForo_Search_Indexer();
			$dataHandler = XenForo_Search_DataHandler_Abstract::create('XenGallery_Search_DataHandler_Media');

			$dataHandler->deleteFromIndex($indexer, $media);
		}

		if ($this->isChanged('media_state'))
		{
			$this->_updateModerationQueue($media);
		}
	}

	/**
	* Post-delete handling.
	*/
	protected function _postDelete()
	{
		if ($this->getExisting('media_state') == 'visible')
		{
			$this->updateUserMediaCount(false);
		}

		$db = $this->_db;

		$mediaId = $db->quote($this->get('media_id'));

		$db->delete('xengallery_comment', "media_id = $mediaId");
		$db->delete('xengallery_rating', "media_id = $mediaId");

		$db->update('xf_attachment', array('unassociated' => 1), 'content_type = \'xengallery_media\' AND content_id = ' . $mediaId);
		
		$indexer = new XenForo_Search_Indexer();
		$dataHandler = XenForo_Search_DataHandler_Abstract::create('XenGallery_Search_DataHandler_Media');

		$dataHandler->deleteFromIndex($indexer, $this->getMergedData());

		$this->_updateDeletionLog(true);
	}

	protected function _updateDeletionLog($hardDelete = false)
	{
		if ($hardDelete
			|| ($this->isChanged('media_state') && $this->getExisting('media_state') == 'deleted')
		)
		{
			$this->getModelFromCache('XenForo_Model_DeletionLog')->removeDeletionLog(
				'xengallery_media', $this->get('media_id')
			);
		}

		if ($this->isChanged('media_state') && $this->get('media_state') == 'deleted')
		{
			$reason = $this->getExtraData(self::DATA_DELETE_REASON);
			$this->getModelFromCache('XenForo_Model_DeletionLog')->logDeletion(
				'xengallery_media', $this->get('media_id'), $reason
			);
		}
	}
	
	protected function _updateModerationQueue(array $media)
	{
		if (!$this->isChanged('media_state'))
		{
			return;
		}

		if ($this->get('media_state') == 'moderated')
		{
			$this->getModelFromCache('XenForo_Model_ModerationQueue')->insertIntoModerationQueue(
				'xengallery_media', $this->get('media_id'), $this->get('media_date')
			);
			
			$indexer = new XenForo_Search_Indexer();
			$dataHandler = XenForo_Search_DataHandler_Abstract::create('XenGallery_Search_DataHandler_Media');

			$dataHandler->deleteFromIndex($indexer, $media);			
		}
		else if ($this->getExisting('media_state') == 'moderated')
		{
			$this->getModelFromCache('XenForo_Model_ModerationQueue')->deleteFromModerationQueue(
				'xengallery_media', $this->get('media_id')
			);
		}
	}	

	public function updateRating($adjustSum = null, $adjustCount = null)
	{
		if ($adjustSum === null && $adjustCount === null)
		{
			$rating = $this->_db->fetchRow('
				SELECT COUNT(*) AS total, SUM(rating) AS sum
				FROM xengallery_rating
				WHERE media_id = ?
				', $this->get('media_id'));

			$this->set('rating_sum', $rating['sum']);
			$this->set('rating_count', $rating['total']);
		}
		else
		{
			if ($adjustSum !== null)
			{
				$this->set('rating_sum', $this->get('rating_sum') + $adjustSum);
			}
			if ($adjustCount !== null)
			{
				$this->set('rating_count', $this->get('rating_count') + $adjustCount);
			}
		}

		if ($this->get('rating_count'))
		{
			$this->set('rating_avg', $this->get('rating_sum') / $this->get('rating_count'));
		}
		else
		{
			$this->set('rating_avg', 0);
		}
	}

	public function updateUserMediaCount($increase = true)
	{
		$value = '+ 1';

		if (!$increase)
		{
			$value = '- 1';
		}

		$this->_db->query("
			UPDATE xf_user
			SET media_count = media_count $value
			WHERE user_id = ?
			", $this->get('user_id'));
	}

	/**
	* @return XenGallery_Model_Media
	*/
	protected function _getMediaModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Media');
	}

	/**
	* @return XenGallery_Model_Category
	*/
	protected function _getCategoryModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Category');
	}        
}