<?php

class XenGallery_Model_Media extends XenForo_Model
{
	const FETCH_USER = 0x01;
	const FETCH_USER_OPTION = 0x02;
	const FETCH_ATTACHMENT = 0x04;
	const FETCH_DELETION_LOG = 0x08;
	const FETCH_NEXT_PREV = 0x10;	
	const FETCH_CATEGORY = 0x20;
	
	// TODO: let these be tunable or dynamic
	// From XenForo Resource Manager
	public static $voteThreshold = 10;
	public static $averageVote = 3;
		
	/**
	 * Gets a single media record specified by its ID
	 *
	 * @param integer $mediaId
	 *
	 * @return array
	 */
	public function getMediaById($mediaId, array $fetchOptions = array())
	{
		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);
		
		$subQuery = '';
		$bind = $mediaId;
		
		if (isset($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_NEXT_PREV)
			{
				$subQuery = ',
					(
						SELECT media.media_id
						FROM xengallery_media AS media WHERE media.media_id < ? AND media.xengallery_category_id = category.xengallery_category_id AND media.media_state = \'visible\' ORDER BY media.media_date DESC, media.media_id DESC LIMIT 1
					) as next_id,
					(
						SELECT media.media_title
						FROM xengallery_media AS media WHERE media.media_id = next_id LIMIT 1
					) as next_title,					
					(
						SELECT media.media_id
						FROM xengallery_media AS media WHERE media.media_id > ? AND media.xengallery_category_id = category.xengallery_category_id AND media.media_state = \'visible\' ORDER BY media.media_date ASC, media.media_id ASC LIMIT 1
					) as prev_id,
					(
						SELECT media.media_title
						FROM xengallery_media AS media WHERE media.media_id = prev_id LIMIT 1
					) as prev_title';
					
				$bind = array($mediaId, $mediaId, $mediaId);
			}
		}

		return $this->_getDb()->fetchRow('
			SELECT media.*
				' . $subQuery . '			
				' . $joinOptions['selectFields'] . '
			FROM xengallery_media AS media
				' . $joinOptions['joinTables'] . '
			WHERE media.media_id = ?
		', $bind);
	}
	
	/**
	 * Gets a single media record specified by its attachment ID
	 *
	 * @param integer $attachmentId
	 *
	 * @return array
	 */
	public function getMediaByAttachmentId($attachmentId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xengallery_media
			WHERE attachment_id = ?
		', $attachmentId);
	}
	
	public function getNextPrevMediaById($mediaId, $next = true, array $fetchOptions = array())
	{			
		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);
		
		return $this->_getDb()->fetchRow('
			SELECT media.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_media AS media
				' . $joinOptions['joinTables'] . '
			WHERE media.media_id = ' . $subClause . '
			AND media.xengallery_category_id = ?
		', $this->getCategoryIdByMediaId($mediaId));
	}
	
	public function getCategoryIdByMediaId($mediaId)
	{
		return $this->_getDb()->fetchOne('
			SELECT xengallery_category_id
			FROM xengallery_media
			WHERE media_id = ?
		', $mediaId);
	}
	
	/**
	 * Gets media records specified by their IDs
	 *
	 * @param array $mediaIds
	 *
	 * @return array
	 */
	public function getMediaByIds($mediaIds, array $fetchOptions = array())
	{
		$db = $this->_getDb();
		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);

		return $this->fetchAllKeyed('
			SELECT media.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_media AS media
				' . $joinOptions['joinTables'] . '
			WHERE media.media_id IN (' . $db->quote($mediaIds) . ')
		', 'media_id');
	}	
	
	/**
	 * Gets a single media record specified by its ID
	 *
	 * @param integer $mediaId
	 *
	 * @return array
	 */
	public function getMedia(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareMediaConditions($conditions, $fetchOptions);
		
		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		$sqlClauses = $this->prepareMediaFetchOptions($fetchOptions);

		$media = $this->fetchAllKeyed($this->limitQueryResults('
			SELECT media.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_media AS media
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				' . $sqlClauses['orderClause'] . '			
			', $limitOptions['limit'], $limitOptions['offset']
		), 'media_id');
		
		return $media;
	}
	
	public function countMedia(array $conditions = array())
	{	
		$fetchOptions = array();
		$whereClause = $this->prepareMediaConditions($conditions, $fetchOptions);

		$joinOptions = $this->prepareMediaFetchOptions($fetchOptions);
				
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xengallery_media AS media
			' . $joinOptions['joinTables'] . '
			WHERE ' . $whereClause
		);
	}	
	
	public function getTopContributors($limit)
	{
		return $this->_getDb()->fetchAll('
			SELECT user.user_id, user.username,
				user.media_count, user.avatar_date,
				user.gravatar, user.avatar_width,
				user.avatar_height, user.display_style_group_id
			FROM xf_user AS user
			ORDER BY media_count DESC
			LIMIT ?
		', $limit);
	}
	
	/**
	 * Gets media IDs in the specified range. The IDs returned will be those immediately
	 * after the "start" value (not including the start), up to the specified limit.
	 *
	 * @param integer $start IDs greater than this will be returned
	 * @param integer $limit Number of media items to return
	 *
	 * @return array List of IDs
	 */
	public function getMediaIdsInRange($start, $limit, $mediaType = 'image_upload')
	{
		$db = $this->_getDb();
		
		$mediaTypes = array(
			'image_upload',
			'video_embed'
		);
		
		if ($mediaType == 'all')
		{
			return $db->fetchCol($db->limit('
				SELECT media_id
				FROM xengallery_media
				WHERE media_id > ?
				AND media_type IN (' . $db->quote($mediaTypes) .')
				ORDER BY media_id
			', $limit), $start);
		}
		else
		{
			return $db->fetchCol($db->limit('
					SELECT media_id
					FROM xengallery_media
					WHERE media_id > ?
					AND media_type = \'image_upload\'
					ORDER BY media_id
				', $limit), $start);			
		}
	}
	
	/**
	 * Gets the IDs of threads that the specified user has not read. Doesn't not work for guests.
	 * Doesn't include deleted or moderated.
	 *
	 * @param integer $userId
	 * @param array $fetchOptions Fetching options
	 *
	 * @return array List of thread IDs
	 */
	public function getUnviewedMediaIds($userId, array $fetchOptions = array())
	{
		if (!$userId)
		{
			return array();
		}
		
		$categoryWhereClause = '';
		if (!empty($fetchOptions['xengallery_category_id']))
		{
			$categoryWhereClause = 'AND xengallery_category_id = ' . $fetchOptions['xengallery_category_id'];
		}

		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

		$autoReadDate = XenForo_Application::$time - (XenForo_Application::get('options')->readMarkingDataLifetime * 86400);

		return $this->_getDb()->fetchCol($this->limitQueryResults(
			'
				SELECT media.media_id
				FROM xengallery_media AS media
				LEFT JOIN xengallery_media_user_view AS media_view ON
					(media_view.media_id = media.media_id AND media_view.user_id = ?
					AND media_view.media_view_date >= media.last_comment_date)
				WHERE media_view.media_view_date IS NULL
					AND media.media_date > ?
					AND media.media_state <> \'moderated\'
					AND media.media_state <> \'deleted\' ' .
					$categoryWhereClause . '
				ORDER BY media.media_date DESC
			', $limitOptions['limit'], $limitOptions['offset']
		), array($userId, $autoReadDate));
	}
	
	/**
	 * Attempts to update any instances of an old username in like_users with a new username
	 *
	 * @param integer $oldUserId
	 * @param integer $newUserId
	 * @param string $oldUsername
	 * @param string $newUsername
	 */
	public function batchUpdateLikeUser($oldUserId, $newUserId, $oldUsername, $newUsername)
	{
		$db = $this->_getDb();
	
		// note that xf_liked_content should have already been updated with $newUserId
	
		$db->query('
			UPDATE (
				SELECT content_id FROM xf_liked_content
				WHERE content_type = \'xengallery_media\'
				AND like_user_id = ?
			) AS temp
			INNER JOIN xengallery_media AS media ON (media.media_id = temp.content_id)
			SET like_users = REPLACE(like_users, ' .
			$db->quote('i:' . $oldUserId . ';s:8:"username";s:' . strlen($oldUsername) . ':"' . $oldUsername . '";') . ', ' .
			$db->quote('i:' . $newUserId . ';s:8:"username";s:' . strlen($newUsername) . ':"' . $newUsername . '";') . ')
		', $newUserId);
	}
	
	/**
	 * Marks the given media as viewed.
	 *
	 * @param array $media Media info
	 * @param array|null $viewingUser
	 *
	 * @return boolean True if marked as viewed
	 */
	public function markMediaViewed(array $media, array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		$userId = $viewingUser['user_id'];
		if (!$userId)
		{
			return false;
		}

		if (!array_key_exists('media_view_date', $media))
		{
			$media['media_view_date'] = $this->getUserMediaViewDate($userId, $media['media_id']);
		}

		$this->_getDb()->query('
			INSERT INTO xengallery_media_user_view
				(user_id, media_id, media_view_date)
			VALUES
				(?, ?, ?)
			ON DUPLICATE KEY UPDATE media_view_date = VALUES(media_view_date)
		', array($userId, $media['media_id'], XenForo_Application::$time));

		return true;
	}
	
	/**
	 * Get the time when a user viewed the given media.
	 *
	 * @param integer $userId
	 * @param integer $mediaId
	 *
	 * @return integer|null Null if guest; timestamp otherwise
	 */
	public function getUserMediaViewDate($userId, $mediaId)
	{
		if (!$userId)
		{
			return null;
		}

		$readDate = $this->_getDb()->fetchOne('
			SELECT media_view_date
			FROM xengallery_media_user_view
			WHERE user_id = ?
				AND media_id = ?
		', array($userId, $mediaId));

		$autoReadDate = XenForo_Application::$time - (XenForo_Application::get('options')->readMarkingDataLifetime * 86400);
		return max($readDate, $autoReadDate);
	}	
	
	public function rebuildUserMediaCounts(array $userIds)
	{
		if (!is_array($userIds))
		{
			return false;
		}
		
		$db = $this->_getDb();
		
		XenForo_Db::beginTransaction($db);
		
		foreach ($userIds AS $userId)
		{
			$mediaCount = $db->fetchOne('
				SELECT COUNT(*)
				FROM xengallery_media
				WHERE user_id = ?
					AND media_state = \'visible\'
			', $userId);
			
			$db->update('xf_user', array('media_count' => $mediaCount), 'user_id = ' . $db->quote($userId));
		}
		
		XenForo_Db::commit($db);
		
		return true;
	}
	
	/**
	 * Gets the average rating based on the sum and count stored.
	 *
	 * @param integer $sum
	 * @param integer $count
	 * @param boolean $round If true, return rating to the nearest 0.5, otherwise full float.
	 *
	 * @return float
	 */
	public function getRatingAverage($sum, $count, $round = false)
	{
		if ($count == 0)
		{
			return 0;
		}

		$average = $sum / $count;

		if ($round)
		{
			$average = round($average / 0.5, 0) * 0.5;
		}

		return $average;
	}
	
	/**
	 * Logs the viewing of a media item.
	 *
	 * @param integer $mediaId
	 */
	public function logMediaView($mediaId)
	{
		$this->_getDb()->query('
			INSERT ' . (XenForo_Application::get('options')->enableInsertDelayed ? 'DELAYED' : '') . ' INTO xengallery_media_view
				(media_id)
			VALUES
				(?)
		', $mediaId);
	}

	/**
	 * Updates media views in bulk.
	 */
	public function updateMediaViews()
	{
		$db = $this->_getDb();

		$updates = $db->fetchPairs('
			SELECT media_id, COUNT(*)
			FROM xengallery_media_view
			GROUP BY media_id
		');

		XenForo_Db::beginTransaction($db);

		$db->query('TRUNCATE TABLE xengallery_media_view');

		foreach ($updates AS $mediaId => $views)
		{
			$db->query('
				UPDATE xengallery_media SET
					view_count = view_count + ?
				WHERE media_id = ?
			', array($views, $mediaId));
		}

		XenForo_Db::commit($db);
	}
	
	
	public function prepareMedia($media)
	{
		if (!empty($media['thumbnail_width']))
		{
			if ($media['thumbnail_width'])
			{
				$media['thumbnailUrl'] = $this->getMediaThumbnailUrl($media);
			}
			else
			{
				$media['thumbnailUrl'] = '';
			}
			
			$media['deleteUrl'] = XenForo_Link::buildPublicLink('attachments/delete', $media);
			$media['viewUrl'] = XenForo_Link::buildPublicLink('attachments', $media);
			
			$media['extension'] = strtolower(substr(strrchr($media['filename'], '.'), 1));
		}		
		
		$visitor = XenForo_Visitor::getInstance();
		
		if (isset($media['user_id']))
		{
			$media['isIgnored'] = array_key_exists($media['user_id'], $visitor->ignoredUsers);
		}
		
		if (isset($media['media_title']))
		{
			$media['media_title'] = XenForo_Helper_String::censorString($media['media_title']);
			$media['media_description'] = XenForo_Helper_String::censorString($media['media_description']);			
		}			

		return $media;
	}	
	
	public function prepareMediaItems($media)
	{
		foreach ($media AS &$_media)
		{
			$_media = $this->prepareMedia($_media);
		}
		
		return $media;
	}

	/**
	 * Prepares join-related fetch options.
	 *
	 * @param array $fetchOptions
	 *
	 * @return array Containing 'selectFields' and 'joinTables' keys.
	 */
	public function prepareMediaFetchOptions(array $fetchOptions)
	{
				
		$selectFields = '';
		$joinTables = '';		
		$orderBy = '';
		
		if (!empty($fetchOptions['order']))
		{
			$orderBySecondary = '';

			switch ($fetchOptions['order'])
			{
				case 'rand':
					$orderBy = 'RAND()';
					break;
					
				case 'media_date':
				case 'new':
				default:
					$orderBy = 'media.media_date';
					break;
					
				case 'media_id':
					$orderBy = 'media.media_date';
					$orderBySecondary = ', media.media_id DESC';
					break;
					
				case 'rating_avg':
					$orderBy = 'media.rating_avg';
					$orderBySecondary = ', media.media_date DESC';
					break;
					
				case 'view_count':
					$orderBy = 'media.view_count';
					$orderBySecondary = ', media.media_date DESC';
					break;
					
				case 'comment_count':
					$orderBy = 'media.comment_count';
					$orderBySecondary = ', media.media_date DESC';
					break;	
					
				case 'rating_count':
					$orderBy = 'media.rating_count';
					$orderBySecondary = ', media.media_date DESC';
					break;											
					
				case 'likes':
					$orderBy = 'media.likes';
					$orderBySecondary = ', media.media_date DESC';
					break;										
			}
			if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'desc')
			{
				$orderBy .= ' DESC';
			}
			else
			{
				$orderBy .= ' ASC';
			}
			
			$orderBy .= $orderBySecondary;
		}

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_CATEGORY)
			{
				$selectFields .= ',
					category.*';
				$joinTables .= '
					LEFT JOIN xengallery_category AS category ON
						(category.xengallery_category_id = media.xengallery_category_id)';
			}

			if ($fetchOptions['join'] & self::FETCH_USER)
			{
				$selectFields .= ',
					user.*, user_profile.*, IF(user.username IS NULL, media.username, user.username) AS username';
				$joinTables .= '
					LEFT JOIN xf_user AS user ON
						(user.user_id = media.user_id)
					LEFT JOIN xf_user_profile AS user_profile ON
						(user_profile.user_id = media.user_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_USER_OPTION)
			{
				$selectFields .= ',
					user_option.*';
				$joinTables .= '
					LEFT JOIN xf_user_option AS user_option ON
						(user_option.user_id = media.user_id)';
			}

            if ($fetchOptions['join'] & self::FETCH_ATTACHMENT)
            {
                $this->getModelFromCache('XenForo_Model_Attachment'); // to get the updated $dataColumns, will not needed with [bd] Attachment Store 0.9.8+

                $selectFields .= ',
                    attachment.attachment_id, attachment.data_id, attachment.attach_date,' . XenForo_Model_Attachment::$dataColumns;
                $joinTables .= '
                    LEFT JOIN xf_attachment AS attachment ON
                        (attachment.content_type = \'xengallery_media\' AND attachment.attachment_id = media.attachment_id)
                    LEFT JOIN xf_attachment_data AS data ON
                        (data.data_id = attachment.data_id)';
            }
			
			if ($fetchOptions['join'] & self::FETCH_DELETION_LOG)
			{
				$selectFields .= ',
					deletion_log.delete_date, deletion_log.delete_reason,
					deletion_log.delete_user_id, deletion_log.delete_username';
				$joinTables .= '
					LEFT JOIN xf_deletion_log AS deletion_log ON
						(deletion_log.content_type = \'xengallery_media\' AND deletion_log.content_id = media.media_id)';
			}
		}

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause' => ($orderBy ? "ORDER BY $orderBy" : '')
		);
	}
	
	/**
	 * Prepares a set of conditions against which to select media items.
	 *
	 * @param array $conditions List of conditions.
	 * @param array $fetchOptions The fetch options that have been provided. May be edited if criteria requires.
	 *
	 * @return string Criteria as SQL for where clause
	 */
	public function prepareMediaConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'media.user_id IN (' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'media.user_id = ' . $db->quote($conditions['user_id']);
			}
		}

		if (!empty($conditions['xengallery_category_id']))
		{
			if (is_array($conditions['xengallery_category_id']))
			{
				$sqlConditions[] = 'media.xengallery_category_id IN (' . $db->quote($conditions['xengallery_category_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'media.xengallery_category_id = ' . $db->quote($conditions['xengallery_category_id']);
			}
		}
		
		if (!empty($conditions['media_id']))
		{
			if (is_array($conditions['media_id']))
			{
				$sqlConditions[] = 'media.media_id IN (' . $db->quote($conditions['media_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'media.media_id = ' . $db->quote($conditions['media_id']);
			}
		}		

		if (isset($conditions['deleted']))
		{
			$sqlConditions[] = $this->prepareStateLimitFromConditions($conditions, 'media', 'media_state');
		}
		else
		{
			// sanity check: only get visible media unless we've explicitly said to get something else
			$sqlConditions[] = "media.media_state = 'visible'";
		}

		return $this->getConditionsForClause($sqlConditions);
	}
	
	public function canViewMedia(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'view'))
		{
			return true;
		}		
		
		$errorPhraseKey = 'xengallery_no_view_permission';
		return false;		
	}
	
	public function canViewMediaItem(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		// Media can be viewed from areas outside the gallery, e.g. profile tabs. This ensures they stay appropriately hidden if they have no permission.
		if (!$this->canViewMedia())
		{
			return false;
		}
		
		if ($media['media_state'] == 'deleted' && !$this->canViewDeletedMedia())
		{
			return false;
		}
		
		if ($media['media_state'] == 'moderated' && !$this->canViewUnapprovedMedia($media))
		{
			return false;
		}

		return true;
	}
	
	public function canViewDeletedMedia(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewDeleted'))
		{
			return true;
		}
	
		return false;		
	}	
	
	public function canLikeMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_you_cannot_like_your_own_media';
			return false;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'like'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_like_permission';
		return false;		
	}
	
	public function canDeleteMedia(array $media, $type = 'soft', &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($type == 'hard')
		{
			return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'hardDeleteAny');
		}

		if ($media['user_id'] == $viewingUser['user_id'])
		{
			return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'delete');
		}

		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny');
	}
	
	public function canEditMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'edit'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_edit_permission';
		return false;		
	}
	
	public function canEditCaption(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCaption'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCaptionAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_caption_edit_permission';
		return false;		
	}	
	
	public function canApproveMedia(array $media, &$errorPhraseKey ='', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $media['media_state'] != 'moderated')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}	
	
	public function canUnapproveMedia(array $media, &$errorPhraseKey ='', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $media['media_state'] != 'visible')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}
	
	public function canViewUnapprovedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}		
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove');
	}	
	
	public function canRateMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($media['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_no_rate_media_by_self';
			return false;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'rate'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_rate_permission';
		return false;
	}	
	
	public function canAddMedia(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'add'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_add_permission';
		return false;
	}
	
	public function canAddMediaToCategory(array $allowedUserGroups, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$this->canAddMedia($errorPhraseKey, $viewingUser))
		{
			$errorPhraseKey = 'xengallery_you_cannot_add_media_to_this_category';
			return false;
		}
		
		if (in_array(-1, $allowedUserGroups))
		{
			return true;
		}
		
		$userModel = $this->getModelFromCache('XenForo_Model_User');
		foreach($allowedUserGroups AS $userGroupId)
		{
			if ($userModel->isMemberOfUserGroup($viewingUser, $userGroupId))
			{
				return true;
			}
		}
		
		$errorPhraseKey = 'xengallery_you_cannot_add_media_to_this_category';
		return false;
	}
	
	/**
	* Checks that the viewing user may managed a reported media item
	*
	* @param array $media
	* @param string $errorPhraseKey
	* @param array $viewingUser
	*
	* @return boolean
	*/
	public function canManageReportedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		return (
			XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny')
		);
	}
	
	/**
	* Checks that the viewing user may manage a moderated media item
	*
	* @param array $media
	* @param string $errorPhraseKey
	* @param array $viewingUser
	*
	* @return boolean
	*/
	public function canManageModeratedMedia(array $media, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		return (
			XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'approveUnapprove')			
		);
	}	
	
	/**
	 * Gets the set of attachment params required to allow uploading.
	 *
	 * @param array $forum
	 * @param array $contentData Information about the content, for URL building
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return array|false
	 */
	public function getAttachmentParams(array $contentData)
	{
		if ($this->canAddMedia())
		{
			return array(
				'hash' => md5(uniqid('', true)),
				'content_type' => 'xengallery_media',
				'content_data' => $contentData
			);
		}
		else
		{
			return false;
		}
	}	
	
	/**
	 * Fetches attachment constraints
	 *
	 * @return array
	 */
	public function getAttachmentConstraints()
	{
		$options = XenForo_Application::get('options');
		
		return array(
			'extensions' => preg_split('/\s+/', trim($options->xengalleryMediaExtensions)),
			'size' => $options->xengalleryMediaMaxFileSize * 1024,
			'width' => $options->xengalleryMaxImageDimensions['width'],
			'height' => $options->xengalleryMaxImageDimensions['height'],
			'count' => $options->xengalleryMediaMaxPerCategory
		);
	}
	
	public function associateAttachmentsAndMedia(array $attachments, array $input, array $category)
	{
		$visitor = XenForo_Visitor::getInstance();
		
		$i = 0;
		foreach ($attachments AS $attachment)
		{		
			$i++;
			
			if (empty($input['media_title'][$attachment['attachment_id']]))
			{
				continue;
			}
				
			$mediaWriter = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			
			$mediaData = array(
				'media_title' => $input['media_title'][$attachment['attachment_id']],
				'media_description' => $input['media_description'][$attachment['attachment_id']],
				'media_state' => $input['media_state'],
				'media_date' => XenForo_Application::$time + $i,
				'xengallery_category_id' => $category['xengallery_category_id'],
				'attachment_id' => $attachment['attachment_id'],
				'user_id' => $visitor->user_id,
				'username' => $visitor->username
			);
			
			$mediaWriter->bulkSet($mediaData);
			$mediaWriter->save();
			
			$db = $this->_getDb();
			XenForo_Db::beginTransaction($db);
			
			$db->update('xf_attachment', array(
					'content_type' => 'xengallery_media',
					'content_id' => $mediaWriter->get('media_id'),
					'temp_hash' => '',
					'unassociated' => 0
			), 'temp_hash = ' . $db->quote($input['attachment_hash']
			));
			
			XenForo_Db::commit($db);
		}
	}
	
	/**
	 * Gets the full path to this attachment's data.
	 *
	 * @param array $data Attachment data info
	 *
	 * @return string
	 */
	public function getMediaDataFilePath(array $data)
	{
		return sprintf('%s/attachments/%d/%d-%s.data',
			XenForo_Helper_File::getInternalDataPath(),
			floor($data['data_id'] / 1000),
			$data['data_id'],
			$data['file_hash']
		);
	}
	
	/**
	 * Gets the full path to this attachment's thumbnail.
	 *
	 * @param string $fileHash Data file hash
	 * @param int $mediaId Media ID
	 *
	 * @return string
	 */
	public function getMediaThumbnailFilePath(array $data)
	{
		return sprintf('%s/xengallery/%d/%d-%s.jpg',
			XenForo_Helper_File::getExternalDataPath(),
			floor($data['data_id'] / 1000),
			$data['data_id'],
			$data['file_hash']
		);
	}
	
	/**
	 * Gets the default file path to the thumbnail directory
	 * 
	 */
	public function getThumbnailFilePath(array $data)
	{
		return sprintf('%s/xengallery/%d/',
			XenForo_Helper_File::getExternalDataPath(),
			floor($data['data_id'] / 1000)
		);
	}
	
	/**
	 * Gets the URL to this media's thumbnail. May be absolute or
	 * relative to the application root directory.
	 *
	 * @param string $fileHash Data file hash
	 * @param int $mediaId Media ID
	 *
	 * @return string
	 */
	public function getMediaThumbnailUrl(array $data)
	{
		return sprintf('%s/xengallery/%d/%d-%s.jpg',
			XenForo_Application::$externalDataUrl,
			floor($data['data_id'] / 1000),
			$data['data_id'],
			$data['file_hash']
		);
	}
}
