<?php

class Brivium_Credits_DataWriter_Action extends XenForo_DataWriter
{
	const DATA_ADDON_ID = 'addOnId';
	/**
	 * Constant for extra data that holds the value for the phrase
	 * that is the title of this link.
	 *
	 * This value is required on inserts.
	 *
	 * @var string
	 */
	const DATA_TITLE = 'phraseTitle';
	const DATA_REVERTED_TITLE = 'phraseRevertedTitle';

	/**
	 * Constant for extra data that holds the value for the phrase
	 * that is the explantion of this action.
	 *
	 * @var string
	 */
	const DATA_EXPLAIN = 'phraseExplain';

	/**
	 * Action that represents whether the action cache will be automatically
	 * rebuilt. Defaults to true.
	 *
	 * @var string
	 */
	const OPTION_REBUILD_CACHE = 'rebuildCache';

	/**
	 * Action that represents whether to validate an action value or simply trust
	 * it as being valid. Trust (false) is generally only used for imports. Defaults to
	 * true.
	 *
	 * @var string
	 */
	const OPTION_VALIDATE_VALUE = 'validateValue';

	/**
	 * Title of the phrase that will be created when a call to set the
	 * existing data fails (when the data doesn't exist).
	 *
	 * @var string
	 */
	protected $_existingDataErrorPhrase = 'requested_action_not_found';

	/**
	 * If this is set, it represents a set of group relations to *replace*.
	 * When it is null, no relations will be updated.
	 *
	 * @var null|array
	 */
	protected $_relations = null;

	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xf_credits_action' => array(
				'action_id'        	=> array('type' => self::TYPE_STRING, 'maxLength' => 100, 'required' => true,
						'verification' => array('$this', '_verifyActionId'), 'requiredError' => 'please_enter_valid_action_id'
				),
				'user_groups'     	=> array('type' => self::TYPE_UNKNOWN, 'verification' => array('$this', '_verifyUserGroups')),
				'forums'     		=> array('type' => self::TYPE_UNKNOWN, 'verification' => array('$this', '_verifyForums')),
				'display_order'     => array('type' => self::TYPE_UINT,   	'default' => 1),
				'amount'    		=> array('type' => self::TYPE_FLOAT,   	'default' => 0),
				'sub_amount'    	=> array('type' => self::TYPE_FLOAT,	'default' => 0),
				'multiplier'    	=> array('type' => self::TYPE_FLOAT,   	'default' => 0),
				'sub_multiplier'    => array('type' => self::TYPE_FLOAT,   	'default' => 0),
				'active'    		=> array('type' => self::TYPE_UINT, 	'default' => 1),
				'alert'    			=> array('type' => self::TYPE_UINT, 	'default' => 1),
				'times'    			=> array('type' => self::TYPE_UINT,    	'default' => 0),
				'max_time'    		=> array('type' => self::TYPE_UINT,    	'default' => 0),
				'target'    		=> array('type' => self::TYPE_STRING, 	'allowedValues' => array('user', 'user_action', 'both'), 'default' => 'user'),
				'apply_max'    		=> array('type' => self::TYPE_UINT,    	'default' => 0),
				'addon_id'    		=> array('type' => self::TYPE_STRING, 	'maxLength' => 25,    'default' => ''),
				'template'    		=> array('type' => self::TYPE_STRING, 'default' => ''),
			)
		);
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$id = $this->_getExistingPrimaryKey($data, 'action_id'))
		{
			return false;
		}
		return array('xf_credits_action' => $this->_getActionModel()->getActionById($id));
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'action_id = ' . $this->_db->quote($this->getExisting('action_id'));
	}

	/**
	 * Gets the default actions for this data writer.
	 */
	protected function _getDefaultActions()
	{
		return array(
			self::OPTION_REBUILD_CACHE => true,
			self::OPTION_VALIDATE_VALUE => true
		);
	}

	/**
	 * Verifies that the action ID contains valid characters and does not already exist.
	 *
	 * @param $actionId
	 *
	 * @return boolean
	 */
	protected function _verifyActionId(&$actionId)
	{
		if (preg_match('/[^a-zA-Z0-9_]/', $actionId))
		{
			$this->error(new XenForo_Phrase('please_enter_an_id_using_only_alphanumeric'), 'action_id');
			
			return false;
		}
		if ($actionId !== $this->getExisting('action_id'))
		{
			if ($this->_getActionModel()->getActionById($actionId))
			{
				$this->error(new XenForo_Phrase('BRC_action_ids_must_be_unique'), 'action_id');
				return false;
			}
		}

		return true;
	}
	
	/**
	 * Verification method for forums
	 *
	 * @param string $serializedData
	 */
	protected function _verifyForums(&$serializedData)
	{
		if ($serializedData === null)
		{
			$serializedData = '';
			return true;
		}
		return XenForo_DataWriter_Helper_Denormalization::verifySerialized($serializedData, $this, 'forums');
	}
	/**
	 * Verification method for user_groups
	 *
	 * @param string $serializedData
	 */
	protected function _verifyUserGroups(&$serializedData)
	{
		if ($serializedData === null)
		{
			$serializedData = '';
			return true;
		}
		return XenForo_DataWriter_Helper_Denormalization::verifySerialized($serializedData, $this, 'user_groups');
	}

	/**
	 * Sets the group relationships for this action.
	 *
	 * @param array $relations List of group relations, format: [group id] => display order.
	 */
	public function setRelations(array $relations)
	{
		$this->_relations = $relations;
	}

	/**
	 * Pre-save handling.
	 */
	protected function _preSave()
	{
		$titlePhrase = $this->getExtraData(self::DATA_TITLE);
		if ($titlePhrase !== null && strlen($titlePhrase) == 0)
		{
			$this->error(new XenForo_Phrase('please_enter_valid_title'), 'title');
		}
	}

	/**
	 * Post-save handling.
	 */
	protected function _postSave()
	{
		$this->_getActionModel()->rebuildActionCache();
	}

	/**
	 * update phrase template
	 */
	public function updateContent()
	{
		$addOnId = $this->getExtraData(self::DATA_ADDON_ID);
		if(!$addOnId)$addOnId = 'Brivium_Credits';
		$titlePhrase = $this->getExtraData(self::DATA_TITLE);
		
		if ($titlePhrase !== null)
		{
			$this->_insertOrUpdateMasterPhrase(
				$this->_getTitlePhraseName($this->get('action_id')), $titlePhrase, $addOnId
			);
		}

		$revertedTitlePhrase = $this->getExtraData(self::DATA_REVERTED_TITLE);
		if ($revertedTitlePhrase !== null)
		{
			$this->_insertOrUpdateMasterPhrase(
				$this->_getRevertedTitlePhraseName($this->get('action_id')), $revertedTitlePhrase, $addOnId
			);
		}

		$explainPhrase = $this->getExtraData(self::DATA_EXPLAIN);
		if ($explainPhrase !== null)
		{
			$this->_insertOrUpdateMasterPhrase(
				$this->_getExplainPhraseName($this->get('action_id')), $explainPhrase, $addOnId
			);
		}
		if($this->isInsert()){
			$data = array(
				'title' => $this->_getActionAlertTemplateName($this->get('action_id')),
				'template' => '',
				'style_id' => 0,
				'addon_id' => $addOnId
			);
			$template = $this->_getTemplateModel()->getTemplateInStyleByTitle($data['title']);
			if(!$template){
				$writer = XenForo_DataWriter::create('XenForo_DataWriter_Template');
				$writer->bulkSet($data);
				$writer->save();
			}
		}
		
	}
	/**
	 * Post-delete handling.
	 */
	protected function _postDelete()
	{
		$this->_getActionModel()->rebuildActionCache();
		$this->_deleteMasterPhrase($this->_getTitlePhraseName($this->get('action_id')));
		$this->_deleteMasterPhrase($this->_getRevertedTitlePhraseName($this->get('action_id')));
		$this->_deleteMasterPhrase($this->_getExplainPhraseName($this->get('action_id')));
	}


	/**
	 * Gets the alert template name for a action.
	 *
	 * @param string $actionId
	 *
	 * @return string
	 */
	protected function _getActionAlertTemplateName($actionId)
	{
		return $this->_getActionModel()->getActionAlertTemplateName($actionId);
	}
	/**
	 * Gets the name of the title phrase for this action.
	 *
	 * @param string $actionId
	 *
	 * @return string
	 */
	protected function _getTitlePhraseName($actionId)
	{
		return $this->_getActionModel()->getActionTitlePhraseName($actionId);
	}
	/**
	 * Gets the name of the reverted title phrase for this action.
	 *
	 * @param string $actionId
	 *
	 * @return string
	 */
	protected function _getRevertedTitlePhraseName($actionId)
	{
		return $this->_getActionModel()->getActionRevertedTitlePhraseName($actionId);
	}

	/**
	 * Gets the name of the explain phrase for this action.
	 *
	 * @param string $actionId
	 *
	 * @return string
	 */
	protected function _getExplainPhraseName($actionId)
	{
		return $this->_getActionModel()->getActionExplainPhraseName($actionId);
	}

	/**
	 * Load action model from cache.
	 *
	 * @return Brivium_Credits_Model_Action
	 */
	protected function _getActionModel()
	{
		return $this->getModelFromCache('Brivium_Credits_Model_Action');
	}
	/**
	 * Lazy load the template model object.
	 *
	 * @return  XenForo_Model_Template
	 */
	protected function _getTemplateModel()
	{
		return $this->getModelFromCache('XenForo_Model_Template');
	}

}